<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Invoice;
use App\Models\Item;
use App\Helpers\InvoicesHelper;
use App\Models\PaymentMethod;
// use Barryvdh\DomPDF\Facade\Pdf;
// use Spatie\Browsershot\Browsershot;
use Dompdf\Dompdf;
use Illuminate\Support\Facades\View;
use Dompdf\Options;
use Auth;

class InvoiceController extends Controller {

    public function __construct() {
		// $this->authorizeResource(Invoice::class, 'invoice');
	}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, ) {

        if(!Auth::user()->can('show invoices')){
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        $invoices = Invoice::query()->where('type','Invoice');

        if (!Auth::user()->can('overview invoices')) {
            $invoices->where('created_by',\Auth::user()->id);
        }

		if (!!$request->trashed) {
			$invoices->withTrashed();
		}

        $invoices = $invoices->with('customer')->get();

        return view('sales.invoices.index', compact('invoices'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create() {

        if (!Auth::user()->can('create invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        $customers = \App\Models\Customer::all();
        $items = Item::all();
        $invoice_no = Invoice::getNextInvNo();
        $payment_methods = PaymentMethod::all();
        $tcs = \App\Models\TC::first()->invoices;

        return view('sales.invoices.create', compact('customers','items','invoice_no','payment_methods','tcs'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, ) {

        $request->validate([ "customer" => "required", "due_date" => "required", "sales_person" => "required", "payment_method" => "required"]);

        try {

            if (!Auth::user()->can('create invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

            $action = $request->action;

            $request->action = ($request->action == "Draft") ? "Draft" : "Completed";

            $InvoicesHelper = new InvoicesHelper;
            $invoice = $InvoicesHelper->saveInvoice($request,"Invoice");

            // $pdf = Pdf::loadView('sales.invoices.templates.1', $invoice->toArray());
            // return $pdf->download('invoice.pdf');
            // // return $invoice;

            if ($action == "Save & Send") {
                return redirect('send_invoice/'.$invoice->invoice_id);
            }

            return redirect()->route('invoices.index', [])->with('success', __('Invoice created successfully.'));
        } catch (\Throwable $e) {
            return redirect()->route('invoices.create', [])->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param \App\Models\Invoice $invoice
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Invoice $invoice,) {
        if (!Auth::user()->can('show invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        return view('sales.invoices.show', compact('invoice'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param \App\Models\Invoice $invoice
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Invoice $invoice,) {
        if (!Auth::user()->can('update invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        $items = Item::all();
        $payment_methods = PaymentMethod::all();

        $customers = \App\Models\Customer::all();

        return view('sales.invoices.edit', compact('invoice', 'customers','payment_methods','items'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Invoice $invoice,) {

        if (!Auth::user()->can('update invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }
        $request->validate([ "customer" => "required", "due_date" => "required", "sales_person" => "required", "payment_method" => "required"]);

        try {

            $InvoicesHelper = new InvoicesHelper;
            $InvoicesHelper->editInvoice($request,$invoice,"Invoice");

            return redirect()->route('invoices.index', [])->with('success', __('Invoice created Updated.'));
        } catch (\Throwable $e) {
            return redirect()->back()->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }

        // $request->validate(["type" => "required", "invoice_no" => "required", "customer_id" => "required", "issue_date" => "required", "due_date" => "required", "shipping_display" => "required", "discount_apply" => "required", "sales_person" => "required", "payment_method_id" => "required", "status" => "required", "created_by" => "required"]);

        // try {
        //     $invoice->type = $request->type;
		// $invoice->project_id = $request->project_id;
		// $invoice->invoice_no = $request->invoice_no;
		// $invoice->customer_id = $request->customer_id;
		// $invoice->issue_date = $request->issue_date;
		// $invoice->subject = $request->subject;
		// $invoice->reference = $request->reference;
		// $invoice->due_date = $request->due_date;
		// $invoice->send_date = $request->send_date;
		// $invoice->ref_number = $request->ref_number;
		// $invoice->shipping_display = $request->shipping_display;
		// $invoice->discount_apply = $request->discount_apply;
		// $invoice->next_invoicing_date = $request->next_invoicing_date;
		// $invoice->paid = $request->paid;
		// $invoice->balance = $request->balance;
		// $invoice->net_amount = $request->net_amount;
		// $invoice->vat = $request->vat;
		// $invoice->gross_amount = $request->gross_amount;
		// $invoice->sales_person = $request->sales_person;
		// $invoice->customer_notes = $request->customer_notes;
		// $invoice->t_c = $request->t_c;
		// $invoice->payment_method_id = $request->payment_method_id;
		// $invoice->status = $request->status;
		// $invoice->created_by = $request->created_by;
        //     $invoice->save();

        //     return redirect()->route('invoices.index', [])->with('success', __('Invoice edited successfully.'));
        // } catch (\Throwable $e) {
        //     return redirect()->route('invoices.edit', compact('invoice'))->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        // }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Models\Invoice $invoice
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Invoice $invoice,) {

        try {
            if (!Auth::user()->can('delete invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }
            \DB::beginTransaction();
            $invoice->payments()->delete();
            $invoice->invoice_items()->delete();
            $invoice->delete();
            \DB::commit();

            return redirect()->route('invoices.index', [])->with('success', __('Invoice deleted successfully'));
        } catch (\Throwable $e) {
            return redirect()->route('invoices.index', [])->with('error', 'Cannot delete Invoice: ' . $e->getMessage());
        }
    }

    //@softdelete

    /**
     * Restore the specified deleted resource from storage.
     *
     * @param \App\Models\Invoice $invoice
     *
     * @return \Illuminate\Http\Response
     */
    public function restore( int $invoice_id,) {
        if (!Auth::user()->can('delete invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        $invoice = Invoice::withTrashed()->find($invoice_id);
        // $this->authorize('delete', [Invoice::class, $invoice]);

        if (!empty($invoice)) {
            \DB::beginTransaction();
            $invoice->restore();
            $invoice->payments()->restore();
            $invoice->invoice_items()->restore();
            \DB::commit();
            return redirect()->route('invoices.index', [])->with('success', __('Invoice restored successfully'));
        } else {
            return redirect()->route('invoices.index', [])->with('error', 'Invoice not found');
        }
    }

    public function purge( int $invoice_id,) {

        if (!Auth::user()->can('delete invoices')) {
            return redirect('dashboard')->with('error',"Permission denied!");
        }

        $invoice = Invoice::withTrashed()->find($invoice_id);
        $this->authorize('delete', [Invoice::class, $invoice]);

        if (!empty($invoice)) {
            $invoice->forceDelete();
            return redirect()->route('invoices.index', [])->with('success', __('Invoice purged successfully'));
        } else {
            return redirect()->route('invoices.index', [])->with('error', __('Invoice not found'));
        }
    }
    //@endsoftdelete

    public function send_invoice_view($id)
    {
        $invoice = Invoice::where('invoice_id',$id)->first();
        $email_body_template = \App\Models\TC::first()->emails;
        $email_body_template = str_replace('{customer_name}', $invoice->customer->name, $email_body_template);
        $email_body_template = str_replace('{user_name}', \Auth::user()->name, $email_body_template);

        return view('sales.invoices.send_invoice',compact('invoice','email_body_template'));
    }

    public function send_invoice(Request $request)
    {
        $request->validate([
            'recipients' => 'required'
            ]);
        try{
            
            $invoice = Invoice::find($request->invoice_id);
            $cc = (($request->cc_receipients)?explode(",",$request->cc_receipients) : "");
            $bcc = (($request->bcc_receipients)?explode(",",$request->bcc_receipients) : "");

            \Mail::to(explode(",",$request->recipients))
                    ->cc($cc)
                    ->bcc($bcc)
                    ->send(new \App\Mail\InvoiceMail($request,$invoice));
                    
            // return $request;

            return redirect('invoices/'.$invoice->id)->with('success','Invoice has been sent');
        }catch(\Throwable $e){
            return redirect()->back()->with('error',$e->getMessage());
        }
    }

    public function export_invoice($type,$id)
    {
        $invoice = Invoice::find($id);
        
        $options = new Options();
        $options->set('defaultFont', 'Courier');
        $options->set('isRemoteEnabled', true);

        $dompdf = new Dompdf($options);
        $view = View::make('sales.invoices.templates.1', compact('invoice'))->render();
        $dompdf->loadHtml($view);

        // Render the HTML as PDF
        $dompdf->render();
        $dompdf->stream('Invoice_'.$id.'.pdf');
    }
}
