<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Personel;
use App\Models\Summary;
use App\Models\Collection;
use App\Models\Branch;
use App\Models\Customer;
use App\Helpers\Restructure;;
use Yajra\DataTables\Facades\DataTables;

class CollectionsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $collections = Collection::query();

            if ($request->date) {
                $collections->whereDate('date', $request->date);
            }

            if ($request->personel_id) {
                if (str_starts_with($request->personel_id, "C")) {
                    $customer_id = str_replace("C", "", $request->personel_id);
                    $collections->where('customer_id', $customer_id);
                } else {
                    $personel_id = str_replace("P", "", $request->personel_id);
                    $collections->where('personel_id', $personel_id);
                }
            }

            if ($request->branch_id) {
                $collections->where('branch_id', $request->branch_id);
            }

            $collections->orderBy('id','desc');

            return DataTables::of($collections)
                ->addColumn('customer_name', function ($row) {
                    return optional($row->customer)->name;
                })
                ->addColumn('personel_name', function ($row) {
                    return optional($row->personel)->name;
                })
                ->addColumn('branch_name', function ($row) {
                    return optional($row->branch)->name;
                })

                ->addColumn('action', function ($row) {
                    if (\Auth::user()->can('edit collections') || ($row->summary->status == "Open")) {
                       return '<a href="'.route('collections.edit', $row->id).'" class="btn btn-sm btn-'.(($row->summary->status == "Open") ? 'primary' : 'danger').'">Edit</a>';
                    }
                    
                })
                ->editColumn('date', function ($collection) {
                    return date('d-M-Y',strtotime($collection->date));
                })
                ->editColumn('amount', function ($collection) {
                    return number_format($collection->amount,2);
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        $branches = Branch::all();
        
        $cpersonels = Personel::all()->map(function($personel) {
            return [
                'id' => "P" . $personel->id,
                'name' => $personel->name,
            ];
        });

        $customers = Customer::all()->map(function($customer) {
            return [
                'id' => "C" . $customer->id,
                'name' => $customer->name,
            ];
        });

        $personels = $customers->merge($cpersonels);

        return view('collections.index', compact('personels', 'branches'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $summary = Summary::find($request->summary_id);$personels = Personel::get();
        $customers = Customer::get();

        $cpersonels = Personel::all()->map(function($personel) {
            return [
                'id' => "P".$personel->id,
                'name' => $personel->name,
            ];
        });

        $customers = Customer::all()->map(function($customer) {
            return [
                'id' => "C".$customer->id,
                'name' => $customer->name,
            ];
        });

        $personels = $customers->merge($cpersonels);
        $collections = Collection::where('summary_id',$summary->id)->get();
        foreach($collections as $collection){
            if($collection->personel_id){
                $collection->collector_id = "P".$collection->personel_id;
            }

            if($collection->customer_id){
                $collection->collector_id = "C".$collection->customer_id;
            }
        }

        // return $collections;
        return view('collections.create',compact('personels','summary','collections'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $summary = Summary::find($request->summary_id);

            $total = 0;

            \DB::beginTransaction();

            $collections = Collection::where('summary_id',$summary->id)->delete();

            foreach($request->items as $item){
                if (gettype($item) == "array") 
                {
                    $item = json_encode($item);
                }

                $item = json_decode($item);

                $amount = str_replace(",", "", $item->amount);

                $customer_id = null;
                $personel_id = null;
                if (str_starts_with($item->personel_id, "C")) {
                    $customer_id = str_replace("C","",$item->personel_id);
                }else{
                    $personel_id = str_replace("P","",$item->personel_id);
                }

                $collection = new Collection;
                $collection->summary_id = $summary->id;
                $collection->branch_id = $summary->branch_id;
                $collection->date = $summary->date;
                $collection->personel_id = $personel_id;
                $collection->customer_id = $customer_id;
                $collection->amount = $amount;
                $collection->reason = $item->reason;
                $collection->created_by_id = \Auth::user()->id;
                $collection->save();

                $total += $collection->amount;
            }

            $summary->cash_sales = $total;
            $summary->save();

            \DB::commit();

            return redirect()->back()->with('success','Collections have been saved!');
        } catch (\Throwable $e) {
            return $e->getMessage();
            return redirect()->back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $cpersonels = Personel::all()->map(function($personel) {
            return [
                'id' => "P" . $personel->id,
                'name' => $personel->name,
            ];
        });

        $customers = Customer::all()->map(function($customer) {
            return [
                'id' => "C" . $customer->id,
                'name' => $customer->name,
            ];
        });

        $personels = $customers->merge($cpersonels);

        $collection = Collection::find($id);
        return view('collections.edit',compact('personels','collection'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'customer_id' => ['required'],
            'amount' => ['required'],
            'reason' => ['required'],
        ]);

        try{
            \DB::beginTransaction();
            $collection = Collection::find($id); // Be sure to validate $id exists
            $amount = str_replace(",", "", $request->amount); // Removes commas from the amount input

            $customer_id = null;
            $personel_id = null;
            if (str_starts_with($request->customer_id, "C")) {
                $customer_id = str_replace("C","",$request->customer_id);
            }else{
                $personel_id = str_replace("P","",$request->customer_id);
            }

            $collectionBefore = Collection::find($id);
            // Update credit details
            $collection->reason = $request->reason;
            $collection->customer_id = $customer_id;
            $collection->personel_id = $personel_id;
            $collection->amount = $amount;
            $collection->save();

            $restructure = new Restructure;

            if ($collection->summary->status == "Closed") {
                if (($collectionBefore->customer_id != $collection->customer_id)) {
                    ($collectionBefore->customer_id)? $restructure->customer_balances($collectionBefore->customer_id) : '';
                    ($collection->customer_id) ? $restructure->customer_balances($collection->customer_id) : '';
                }

                if (($collectionBefore->amount != $collection->amount)) {
                    $restructure->branch_balances($collection->branch_id,$collection->summary_id);
                }
            }

            \DB::commit();

            return redirect()->back()->with('success','Collection Details have been updated!');;
        }catch(\Throwable $ex){
            \DB::rollback();
            return redirect()->back()->withErrors(['error' => $ex->getMessage().' on line '.$ex->getLine()])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
