<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Summary;
use App\Models\Inventory;
use App\Models\Customer;
use App\Models\Discount;
use App\Models\Branch;
use Yajra\DataTables\Facades\DataTables;

class DiscountsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $discounts = Discount::query();

            if ($request->date) {
                $discounts->whereDate('date', $request->date);
            }

            if ($request->customer_id) {
                $discounts->where('customer_id', $request->customer_id);
            }

            if ($request->branch_id) {
                $discounts->whereHas('summary', function($query) use($request){
                    $query->where('branch_id', $request->branch_id);
                });
            }

            return DataTables::of($discounts)
                ->editColumn('date', function ($discount) {
                    return \Carbon\Carbon::parse($discount->date)->format('d M, Y');
                })
                ->addColumn('branch_name', function ($discount) {
                    return $discount->summary->branch->name ?? '-';
                })
                ->addColumn('customer_name', function ($discount) {
                    return $discount->customer->name ?? '-';
                })
                ->addColumn('inventory_name', function ($discount) {
                    return $discount->inventory->name ?? '-';
                })
                ->addColumn('discount_rate', function ($discount) {
                    return $discount->discount_rate ? number_format($discount->discount_rate) : 0;
                })
                ->addColumn('quantity', function ($discount) {
                    return $discount->quantity ? number_format($discount->quantity) : 0;
                })
                ->addColumn('total_discount', function ($discount) {
                    return $discount->total_discount ? number_format($discount->total_discount) : 0;
                })
                ->make(true);
        }

        $branches = Branch::all();
        $customers = Customer::all();
        return view('discounts.index', compact('customers', 'branches'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $inventories = Inventory::all();
        $summary = Summary::find($request->summary_id);
        $customers = Customer::all();
        $discounts = Discount::where('summary_id',$summary->id)->get();

        return view('discounts.create',compact('summary','inventories','customers','discounts'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $summary = Summary::find($request->summary_id);

            $total = 0;
            \DB::beginTransaction();

            Discount::where('summary_id',$summary->id)->delete();

            foreach ($request->items as $item) {
                if (gettype($item) == "array") 
                    {
                        $item = json_encode($item);
                    }

                $item = json_decode($item);

                $rate = str_replace(",", "", $item->rate);
                $item->price = str_replace(",", "", $item->price);

                $inventory = Inventory::find($item->inventory_id);

                $discount = new Discount;
                $discount->summary_id = $summary->id;
                $discount->customer_id = $item->customer_id;
                $discount->inventory_id = $item->inventory_id;
                $discount->date = $summary->date;
                $discount->rate = $inventory->rate;
                $discount->price = $item->price;
                $discount->discount_rate = $rate;
                $discount->quantity = $item->quantity;
                $discount->total_discount = $discount->discount_rate * $discount->quantity;
                $discount->created_by_id = \Auth::user()->id;
                $discount->save();

                $total += $discount->total_discount;
            }

            $summary->discounts = $total;
            $summary->save();

            \DB::commit();

            return redirect()->back()->with('success','Discounts saved successfully!');
        }catch(\Throwable $ex){
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
