<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\GeneralExpense;
use App\Models\PaymentMethod;
use App\Models\Supplier;
use App\Models\Ledger;
use App\Models\Payment;
use App\Models\ExpenseItem;

class GeneralExpensesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $expenses = GeneralExpense::with(['supplier', 'expenseItem'])->latest()->get();
        return view('general_expenses.index', compact('expenses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $suppliers = Supplier::orderBy('name','asc')->get();
        $items = ExpenseItem::orderBy('name','asc')->get();
        $methods = PaymentMethod::orderBy('name','asc')->get();
        return view('general_expenses.create', compact('suppliers', 'items', 'methods'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'expense_date' => 'required',
            'supplier_id' => 'required',
            'expense_item_id' => 'required',
            'payment_method_id' => 'required',
            'amount' => 'required|numeric',
            // 'paid_amount' => 'required|numeric',
            // 'status' => 'required',
        ]);

        $request['amount'] = str_replace(",", "", $request['amount']);
        $payment_method = PaymentMethod::find($request->payment_method_id);

        $paid = 0;

        if ($payment_method->type != "Credit") {
            $paid = $request['amount'];
        }

        $request['paid_amount'] = $paid;

        $request['balance'] = $request->amount - $paid;
        $request['status'] = "Pending";
        \DB::beginTransaction();

        $expense = GeneralExpense::create($request->all());

        $balance = 0;

        $last_ledger_entry = Ledger::where('supplier_id', $request->supplier_id)->orderBy('id', 'desc')->first();
        if ($last_ledger_entry) {
            $balance = $last_ledger_entry->balance;
        }

        if ($payment_method->type != "Credit") {
            $new_balance = $balance;
            $credit_amount = $paid;
            $debit_amount = $paid;

            $payment = new Payment;
            $payment->supplier_id = $request->supplier_id;
            $payment->customer_id = null;
            $payment->amount = $request['amount'];
            $payment->general_expense_id = $expense->id;
            $payment->status = "Pending";
            $payment->date = $request->expense_date;
            $payment->balance = $new_balance;
            $payment->save();
        }else{
            $new_balance = $balance + $request['amount'];
            $credit_amount = null;
            $debit_amount = $request['amount'];
        }

        $new_ledger_entry = new Ledger;
        $new_ledger_entry->customer_id = null;
        $new_ledger_entry->supplier_id = $request->supplier_id;
        $new_ledger_entry->credit = $credit_amount;
        $new_ledger_entry->debit = $debit_amount;
        $new_ledger_entry->label = "General Expense";
        $new_ledger_entry->general_expense_id = $expense->id;
        $new_ledger_entry->balance = $new_balance; 
        $new_ledger_entry->date = $expense->expense_date;
        $new_ledger_entry->save();

        $supplier = Supplier::find($request->supplier_id);
        $supplier->balance = $new_balance;
        $supplier->save();

        \DB::commit();

        return redirect()->route('general_expenses.index')->with('success', 'Expense added.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        return view('general_expenses.show', compact('general_expense'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $suppliers = Supplier::all();
        $items = ExpenseItem::all();
        $methods = PaymentMethod::all();
        $general_expense = GeneralExpense::find($id);
        return view('general_expenses.edit', compact('general_expense', 'suppliers', 'items', 'methods'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'expense_date' => 'required',
            'supplier_id' => 'required',
            'expense_item_id' => 'required',
            'payment_method_id' => 'required',
            'amount' => 'required|numeric',
            // 'paid_amount' => 'required|numeric',
            // 'status' => 'required',
        ]);

        $request['balance'] = $request->amount - $request->paid_amount;

        $request['amount'] = str_replace(",", "", $request['amount']);
        $payment_method = PaymentMethod::find($request->payment_method_id);

        $paid = 0;

        if ($payment_method->type != "Credit") {
            $paid = $request['amount'];
        }

        $request['paid_amount'] = $paid;

        $request['balance'] = $request->amount - $paid;
        $request['status'] = "Pending";

        $general_expense = GeneralExpense::find($id);

        $general_expense->update($request->all());

        return redirect()->route('general_expenses.index')->with('success', 'Expense updated.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $general_expense = GeneralExpense::find($id);
        $general_expense->delete();
        return redirect()->route('general_expenses.index')->with('success', 'Expense deleted.');
    }
}
