<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Ledger;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Helpers\Restructure;

class PaymentsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        try {
            $supplier_id = null;
            $customer_id = null;

            if ($request->customer_id) {
                $customer_id = $request->customer_id;
            }

            if($request->supplier_id){
                $supplier_id = $request->supplier_id;
            }

            $customers = Customer::all();
            $suppliers = Supplier::all();
            $payment_methods = PaymentMethod::all();
            

            return view('payments.create',compact('supplier_id','customer_id','customers','suppliers','payment_methods'));
        } catch (\Throwable $e) {
            return $e->getMessage();
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'nullable|exists:customers,id|required_without:supplier_id',
            'supplier_id' => 'nullable|exists:suppliers,id|required_without:customer_id',
            'amount' => 'required|min:0.01',
        ]);

        try{
            \DB::beginTransaction();

            $request->amount = str_replace(',', '', $request->amount);

            $payment = new Payment;
            $payment->supplier_id = $request->supplier_id;
            $payment->customer_id = $request->customer_id;
            $payment->amount = $request->amount;
            $payment->reference = $request->reference;
            $payment->date = $request->date;
            $payment->payment_method_id = $request->payment_method_id;

            $balance = 0;

            if ($request->customer_id) {
                $last_ledger_entry = Ledger::where('customer_id', $request->customer_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }
            }

            if ($request->supplier_id) {
                $last_ledger_entry = Ledger::where('supplier_id', $request->supplier_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }
            }

            $new_balance = $balance - $request->amount;

            $payment->balance = $new_balance;
            $payment->status = "Pending";

            $payment->save();

            $new_ledger_entry = new Ledger;
            $new_ledger_entry->customer_id = $request->customer_id;
            $new_ledger_entry->supplier_id = $request->supplier_id;
            $new_ledger_entry->credit = $request->amount;
            $new_ledger_entry->label = "Payment";
            $new_ledger_entry->payment_id = $payment->id;
            $new_ledger_entry->balance = $new_balance; 
            $new_ledger_entry->date = $request->date;
            $new_ledger_entry->save();

            if ($request->supplier_id) {
                $supplier = Supplier::find($request->supplier_id);
                $supplier->balance = $new_balance;
                $supplier->save();
            }

            if ($request->customer_id) {
                $customer = Customer::find($request->customer_id);
                $customer->balance = $new_balance;
                $customer->save();
            }

            \DB::commit();
            return redirect()->back()->with('success', 'Payment added successfully.');
        }catch(\Throwable $ex){
            \DB::rollback();
            // return $ex->getMessage();
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $payment = Payment::find($id);
        $customers = Customer::all();
        $suppliers = Supplier::all();
        $payment_methods = PaymentMethod::all();

        return view('payments.edit',compact('payment','customers','suppliers','payment_methods'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'customer_id' => 'nullable|exists:customers,id|required_without:supplier_id',
            'supplier_id' => 'nullable|exists:suppliers,id|required_without:customer_id',
            'amount' => 'required|min:0.01',
        ]);

        try{
            \DB::beginTransaction();

            $request->amount = str_replace(',', '', $request->amount);

            $old_payment = Payment::find($id);

            $payment = Payment::find($id);
            $payment->supplier_id = $request->supplier_id;
            $payment->customer_id = $request->customer_id;
            $payment->amount = $request->amount;
            $payment->reference = $request->reference;
            $payment->payment_method_id = $request->payment_method_id;
            $payment->status = "Pending";

            $balance = 0;

            if ($request->customer_id) {
                $last_ledger_entry = Ledger::where('customer_id', $request->customer_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }
            }

            if ($request->supplier_id) {
                $last_ledger_entry = Ledger::where('supplier_id', $request->supplier_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }
            }

            $new_balance = $balance - $request->amount;

            $payment->balance = $new_balance;

            $payment->save();
            $restructure = new Restructure;

            if ($request->customer_id) {
                
                if ($old_payment->amount != $payment->amount) {
                    $restructure->customer_balances($payment->customer_id);
                }

                if ($old_payment->customer_id != $payment->customer_id) {
                    $restructure->customer_balances($old_payment->customer_id);
                    $restructure->customer_balances($payment->customer_id);
                }
            }else{

                if ($old_payment->amount != $payment->amount) {
                    $restructure->supplier_balances($payment->supplier_id);
                }

                if ($old_payment->supplier_id != $payment->supplier_id) {
                    $restructure->supplier_balances($old_payment->supplier_id);
                    $restructure->supplier_balances($payment->supplier_id);
                }
            }

            \DB::commit();
            return redirect()->back()->with('success', 'Payment added successfully.');
        }catch(\Throwable $ex){
            \DB::rollback();
            // return $ex->getMessage();
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
