<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Summary;
use App\Models\DaySummary;
use App\Models\Branch;
use App\Models\Ledger;
use App\Models\Payment;
use App\Models\Personel;
use App\Models\Customer;
use App\Models\Collection;
use App\Models\Credit;
use App\Models\Supplier;
use App\Models\BusinessSummary;
use Yajra\DataTables\Facades\DataTables;

class SummariesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // $summaries = Summary::all();

        if ($request->ajax()) {
            $data = Summary::orderBy('id','desc')->get();

            return DataTables::of($data)
                    ->editColumn('status', function($row) {
                            return ucfirst($row->status); // Format the status column
                })
                ->addColumn('branch_name', function ($row) {
                        return $row?->branch?->name ?? '-';
                })
                ->addColumn('personel', function ($row) {
                        return $row?->personel?->name ?? '-';
                })
                ->addColumn('user', function ($row) {
                        return $row?->user?->name ?? '-';
                })
                ->editColumn('opening_balance', function ($row) {
                    return number_format($row->opening_balance,2);
                })
                ->editColumn('closing_balance', function ($row) {
                    return number_format($row->closing_balance,2);
                })
                ->addColumn('actions', function($row){
                    $action = "";
                    if ($row->status == "Closed") {
                        $action = '<a class="btn btn-primary btn-sm view" href="'.route('summaries.show',$row->id).'">Show</a>';
                    }else{
                        $action = '<a href="'.url('summaries/modules/'.$row->id).'" class="btn btn-danger btn-sm">Continue</a>';
                    }
                    return $action;
                })
                ->rawColumns(['actions'])
                ->make(true);
        }


        return view('summaries.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $branches = Branch::get();
        $personels = Personel::all();
        return view('summaries.create',compact('branches','personels'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'branch_id' => 'required',
            'date' => 'required',
            'personel_id' => 'required',
        ]);

        try {
            // $summary = Summary::whereDate('date', '>=', $request->date)->where('branch_id',$request->branch_id)->first();

            // if ($summary) {
            //     return redirect()->back()->withInput()->withErrors(['error'=>'Branch Summary already exists with the same or after date!']);
            // }

            $opening_balance = 0;

            $last_summary = Summary::whereDate('date', '<', $request->date)->where('branch_id',$request->branch_id)->orderBy('date','desc')->first();

            if ($last_summary) {
                $opening_balance = $last_summary->closing_balance;
            }

            \DB::beginTransaction();
            $summary = new Summary;
            $summary->branch_id = $request->branch_id;
            $summary->date = $request->date;
            $summary->opening_balance = $opening_balance;
            $summary->closing_balance = 0;
            $summary->profits = 0;
            $summary->transfers = 0;
            $summary->total_sales = 0;
            $summary->intake = 0;
            $summary->credit_sales = 0;
            $summary->discounts = 0;
            $summary->expenses = 0;
            $summary->profits_total = 0;
            $summary->cash_sales = 0;
            $summary->stage = "Collections";
            $summary->user_id = \Auth::user()->id;
            $summary->personel_id = $request->personel_id;
            $summary->save();

            \DB::commit();

            return redirect('collections/create?summary_id='.$summary->id);
        } catch (\Throwable $e) {
            return $e->getMessage();
            return redirect()->back()->withErrors(['error'=>$e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $summary = Summary::where('id',$id)->with('collections','credits','summary_discounts','intakes','summary_transfers','summary_expenses')->first();
        return view('summaries.show',compact('summary'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $summary = Summary::find($id);

        $summary->delete();

        return redirect('summaries')->with('success','Summary has been deleted!');
    }

    public function summary($summary_id)
    {
        $summary = Summary::where('id',$summary_id)
                    ->with('collections','credits','summary_discounts','intakes','summary_transfers','summary_expenses')
                    ->withSum('collections','amount')
                    ->withSum('credits','amount')
                    ->withSum('summary_discounts','total_discount')
                    ->withSum('intakes','amount')
                    ->withSum('summary_transfers','amount')
                    ->withSum('summary_expenses','amount')
                    ->withSum('recoveries','amount')
                    ->withSum('summary_profits','amount')
                    ->first();

        $last_summary = Summary::whereDate('date', '<=', $summary->date)
                                ->where('id','!=',$summary->id)
                                ->where('branch_id',$summary->branch_id)
                                ->orderBy('date', 'desc')->first();

        $opening_balance = 0;
        if ($last_summary) {
            $opening_balance = $last_summary->closing_balance;
        }else{
            $opening_balance = $summary->opening_balance;
        }

        // dd($summary);

        $summarisation = new \stdClass;
        $summarisation->opening_balance = $opening_balance;
        $summarisation->sales = $summary->collections_sum_amount + $summary->credits_sum_amount + $summary->summary_discounts_sum_total_discount + $summary->summary_expenses_sum_amount;
        $summarisation->intakes = $summary->intakes_sum_amount;
        $summarisation->transfers = $summary->summary_transfers_sum_amount;
        $summarisation->profits = $summary->summary_profits_sum_amount;
        $summarisation->closing_balance = ($opening_balance + $summarisation->intakes + $summarisation->profits) - ($summarisation->sales + $summarisation->transfers);
        // $summary->opening_balance + $summary->intake+$summary->profits_total - ($summary->total_sales + $summary->transfers);
        return view('summaries.summary',compact('summary','summarisation'));
    }

    public function complete_summary(Request $request)
    {
        try{

            $summary = Summary::find($request->summary_id);

            $last_summary = Summary::whereDate('date', '<=', $summary->date)
                                ->where('id','!=',$summary->id)
                                ->where('branch_id',$summary->branch_id)
                                ->orderBy('date', 'desc')->first();

            $opening_balance = 0;

            if ($last_summary && $last_summary->status == "Open") {

                return redirect()->back()->withInput()->withErrors(['error'=>'Branch has an open summary!']);
            }

            if ($last_summary) {
               $opening_balance = $last_summary->closing_balance;
            }else{
                $opening_balance = $summary->opening_balance;
            }

            \DB::beginTransaction();
            $summary->opening_balance = $opening_balance;
            $summary->total_sales = $summary->cash_sales + $summary->credit_sales + $summary->expenses + $summary->discounts;
            $summary->status = 'Closed';
            $summary->closing_balance = $summary->opening_balance + $summary->intake+$summary->profits_total - ($summary->total_sales + $summary->transfers);
            $summary->user_id = \Auth::user()->id;
            $summary->save();

            foreach ($summary->collections as $collection) {
                if ($collection->customer_id) {
                    $customer = Customer::find($collection->customer_id);

                    if (!$customer) {
                        \DB::rollback();
                        // return 'Customer with id '.$credit->customer_id.' in collections no longer exists!';
                        return redirect()->back()->withErrors(['success'=>'Customer with id '.$credit->customer_id.' no longer exists!']);
                    }

                    $ledger = new Ledger;
                    $ledger->label = "Credit";
                    $ledger->customer_id = $collection->customer_id;
                    $ledger->debit = $collection->amount;
                    $ledger->date = $summary->date;
                    $ledger->balance = $customer->balance + $collection->amount;
                    $ledger->collection_id = $collection->id;
                    $ledger->status = "Approved";
                    $ledger->save();

                    $customer->balance = $ledger->balance;
                    $customer->save();
                }
            }

            foreach ($summary->credits as $credit) {
                $customer = Customer::find($credit->customer_id);

                if (!$customer) {
                    \DB::rollback();
                    // return 'Customer with id '.$credit->customer_id.' in credits no longer exists!';
                    return redirect()->back()->withErrors(['error'=>'Customer with id '.$credit->customer_id.' no longer exists!']);
                }

                $ledger = new Ledger;
                $ledger->label = "Sale";
                $ledger->customer_id = $credit->customer_id;
                $ledger->debit = $credit->amount;
                $ledger->date = $summary->date;
                $ledger->balance = $customer->balance + $credit->amount;
                $ledger->credit_id = $credit->id;
                $ledger->status = "Approved";
                $ledger->save();

                $customer->balance = $ledger->balance;
                $customer->save();
            }

            foreach ($summary->intakes as $intake) {
                if ($intake->supplier_id) {
                    $supplier = Supplier::find($intake->supplier_id);

                    if (!$supplier) {
                        \DB::rollback();
                        // return 'Supplier with id '.$intake->supplier_id.' in intakes no longer exists!';
                        return redirect()->back()->withInput()->withErrors(['error'=>'Supplier with id '.$intake->supplier_id.' no longer exists!']);
                    }

                    $amount = $intake->quantity * $intake->supplier_rate;

                    $ledger = new Ledger;
                    $ledger->label = "Purchase";
                    $ledger->supplier_id = $intake->supplier_id;
                    $ledger->debit = $amount;
                    $ledger->date = $summary->date;
                    $ledger->balance = $supplier->balance + $amount;
                    $ledger->intake_id = $intake->id;
                    $ledger->status = "Approved";
                    $ledger->save();

                    $supplier->balance = $ledger->balance;
                    $supplier->save();
                }
            }

            foreach($summary->recoveries as $recovery){
                $payment = new Payment;
                $payment->supplier_id = null;
                $payment->customer_id = $recovery->customer_id;
                $payment->amount = $recovery->amount;
                $payment->recovery_id = $recovery->id;
                $payment->disbursement_id = null;
                $payment->reference = "Registered";
                $payment->date = $recovery->date;
                $payment->status = "Pending";

                $balance = 0;
                $last_ledger_entry = Ledger::where('customer_id', $recovery->customer_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }

                $new_balance = $balance - $recovery->amount;
                $payment->balance = $new_balance;
                $payment->save();

                $new_ledger_entry = new Ledger;
                $new_ledger_entry->customer_id = $recovery->customer_id;
                $new_ledger_entry->supplier_id = null;
                $new_ledger_entry->credit = $recovery->amount;
                $new_ledger_entry->label = "Recovery";
                $new_ledger_entry->payment_id = $payment->id;
                $new_ledger_entry->balance = $new_balance; 
                $new_ledger_entry->date = $payment->date;
                $new_ledger_entry->save();

                $customer = Customer::find($recovery->customer_id);
                $customer->balance = $new_balance;
                $customer->save();

                $recovery->status = "Pending";
                $recovery->save();
            }

            $business_summary = BusinessSummary::whereDate('date', $summary->date)->first();

            if (!$business_summary) {
                $business_summary = new BusinessSummary;
                $business_summary->date = $summary->date;
                $business_summary->collections = 0;
                $business_summary->withdrawals = 0;
                $business_summary->balance = 0;
                $business_summary->save();
            }

            $business_summary->collections = Collection::whereNull('customer_id')->where('summary_id',$summary->id)->sum('amount');
            $business_summary->balance = $business_summary->collections - $business_summary->withdrawals;
            $business_summary->save();

            $daySummary = $this->saveDaySummary($summary);

            if (!$daySummary->success) {
                \DB::rollback();
                return redirect()->back()->withErrors(['error' => 'Unable to save day summary!'])->withInput();
            }

            \DB::commit();

            // $recipient = 'musisifred7@gmail.com';
            // $subject = 'Day summary closed';
            // $content = 'Hello,

            // Branch:. '.$summary->branch?->name.',
            // Openning Balance: '.number_format($summary->opening_balance).',
            // Cash Sales: '.number_format($summary->cash_sales).',
            // Closing balance: '.number_format($summary->closing_balance).',

            // Closed By: '.\Auth::user()->name.',';

            // \Mail::raw($content, function ($message) use ($recipient, $subject) {
            //     $message->to($recipient)->subject($subject);
            // });

            return redirect('summaries/'.$summary->id)->with('success','Summary has been completed!');
        }catch(\Throwable $ex){
            // return $ex->getMessage()." on line ".$ex->getLine();
            \DB::rollback();
            return redirect()->back()->withErrors(['error' => $ex->getMessage()." - ".$ex->getLine()])->withInput();
        }
    }

    public function populateDaySummary(){
        $summaries = Summary::where('status','Closed')->orderBy('id','asc')->get();

        foreach ($summaries as $summary) {
            $this->saveDaySummary($summary);
        }
    }
    public function saveDaySummary($summary){
        $response = new \stdClass;
        $response->success = false;
        $response->message = "Unable to save day summary";

        $DaySummary = DaySummary::where('date',$summary->date)->first();

        if (!$DaySummary) {
            $DaySummary = new DaySummary;
            $DaySummary->date = $summary->date;
            $DaySummary->collections = 0;
            $DaySummary->save();
        }

        $collections = \App\Models\Collection::whereNull('customer_id')->where('date',$summary->date)->sum('amount');

        $DaySummary->opening_balance += $summary->opening_balance;
        $DaySummary->closing_balance += $summary->closing_balance;
        $DaySummary->profits += $summary->profits_total;
        $DaySummary->transfers += $summary->transfers;
        $DaySummary->total_sales += $summary->total_sales;
        $DaySummary->intake += $summary->intake;
        $DaySummary->credit_sales += $summary->credit_sales;
        $DaySummary->discounts += $summary->discounts;
        $DaySummary->expenses += $summary->expenses;
        $DaySummary->cash_sales += $summary->cash_sales;
        $DaySummary->recoveries += $summary->total_recoveries;
        $DaySummary->collections += ($collections + $summary->total_recoveries);
        $DaySummary->save();

        $response->success = true;
        $response->message = "successfully";
        return $response;
    }

    public function modules($summary_id)
    {
        $summary = Summary::withSum('collections','amount')
                                ->withSum('credits','amount')
                                ->withSum('summary_discounts','total_discount')
                                ->withSum('intakes','amount')
                                ->withSum('summary_transfers','amount')
                                ->withSum('summary_expenses','amount')
                                ->withSum('recoveries','amount')
                                ->withSum('summary_profits','amount')
                                ->find($summary_id);

        // return $summary;

        return view('summaries.modules',compact('summary'));
    }
}
