<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Supplier;
use App\Models\Ledger;
use App\Helpers\Restructure;
use Illuminate\Support\Facades\View;
use Dompdf\Options;
use Dompdf\Dompdf;

class SupplierController extends Controller {

    public function __construct() {
		// $this->authorizeResource(Supplier::class, 'supplier');
	}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, ) {

        $suppliers = Supplier::query();

		if (!!$request->trashed) {
			$suppliers->withTrashed();
		}

        if(!empty($request->search)) {
			$suppliers->where('name', 'like', '%' . $request->search . '%');
		}

        $suppliers = $suppliers->get();

        return view('suppliers.index', compact('suppliers'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create() {

        return view('suppliers.create', []);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, ) {

        $request->validate(["name" => "required", "contact" => "required", "balance" => "required"]);

        try {

             if ($request->balance) {
                $request->balance  = str_replace(',', '', $request->balance);
            }
            $supplier = new Supplier();
            $supplier->name = $request->name;
    		$supplier->contact = $request->contact;
    		$supplier->alt_contact = $request->alt_contact;
    		$supplier->balance = $request->balance;
            $supplier->save();

            if ($supplier->balance) {
                $ledger = new Ledger;
                $ledger->label = "Opening balance";
                $ledger->supplier_id = $supplier->id;
                $ledger->debit = $supplier->balance;
                $ledger->date = date('Y-m-d');
                $ledger->balance = $supplier->balance;
                $ledger->save();
            }

            return redirect()->route('suppliers.index', [])->with('success', __('Supplier created successfully.'));
        } catch (\Throwable $e) {
            return redirect()->route('suppliers.create', [])->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param \App\Models\Supplier $supplier
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Supplier $supplier,) {

        return view('suppliers.show', compact('supplier'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param \App\Models\Supplier $supplier
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Supplier $supplier,) {

        return view('suppliers.edit', compact('supplier'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Supplier $supplier,) {

        $request->validate(["name" => "required", "contact" => "required", "balance" => "required"]);

        try {
            if ($request->balance) {
                $request->balance  = str_replace(',', '', $request->balance);
            }
            $supplier->name = $request->name;
    		$supplier->contact = $request->contact;
    		$supplier->alt_contact = $request->alt_contact;
    		$supplier->balance = $request->balance;

            $supplier->save();
            if ($request->balance) {
                $restructure = new Restructure;
                $restructure->supplier_balances($supplier->id,$request->balance);
            }

            return redirect()->route('suppliers.index', [])->with('success', __('Supplier edited successfully.'));
        } catch (\Throwable $e) {
            return redirect()->route('suppliers.edit', compact('supplier'))->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Models\Supplier $supplier
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Supplier $supplier,) {

        try {
            $supplier->delete();

            return redirect()->route('suppliers.index', [])->with('success', __('Supplier deleted successfully'));
        } catch (\Throwable $e) {
            return redirect()->route('suppliers.index', [])->with('error', 'Cannot delete Supplier: ' . $e->getMessage());
        }
    }

    //@softdelete

    /**
     * Restore the specified deleted resource from storage.
     *
     * @param \App\Models\Supplier $supplier
     *
     * @return \Illuminate\Http\Response
     */
    public function restore( int $supplier_id,) {

        $supplier = Supplier::withTrashed()->find($supplier_id);
        // $this->authorize('delete', [Supplier::class, $supplier]);

        if (!empty($supplier)) {
            $supplier->restore();
            return redirect()->route('suppliers.index', [])->with('success', __('Supplier restored successfully'));
        } else {
            return redirect()->route('suppliers.index', [])->with('error', 'Supplier not found');
        }
    }

    public function purge( int $supplier_id,) {

        $supplier = Supplier::withTrashed()->find($supplier_id);
        // $this->authorize('delete', [Supplier::class, $supplier]);

        if (!empty($supplier)) {
            $supplier->forceDelete();
            return redirect()->route('suppliers.index', [])->with('success', __('Supplier purged successfully'));
        } else {
            return redirect()->route('suppliers.index', [])->with('error', __('Supplier not found'));
        }
    }
    //@endsoftdelete

    public function download($value='')
    {
        $options = new Options();
        $options->set('defaultFont', 'Courier');
        $options->set('isRemoteEnabled', true);
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isPhpEnabled', true);

        $dompdf = new Dompdf($options);

        $suppliers = Supplier::orderBy('balance','asc')->get();

        // return $suppliers;
            
        $dompdf->loadHtml(view("suppliers.download", compact('suppliers'))->render());

        // Render the HTML as PDF
        $dompdf->render();
        $dompdf->stream('Suppliers.pdf');
    }

    public function ledger(Request $request)
    {
        $entries = null;
        $supplier_id = null;

        if ($request->supplier_id) {
            $supplier_id = $request->supplier_id;
            // $entries = Ledger::where('customer_id',$customer_id)->orderBy('id','asc')->get();

            // foreach ($entries as $entry) {
            //     if ($entry->credit_id) {
            //         $entry->module = "Credit";
            //     }elseif ($entry->collection_id) {
            //         $entry->module = "Collection";
            //     }elseif ($entry->intake_id) {
            //         $entry->module = "Intake";
            //     }elseif($entry->payment_id){
            //         $entry->module = "Payment";
            //     }else{
            //         $entry->module = "Others";
            //     }
            // }

            $entries = Ledger::where('supplier_id', $supplier_id)
                ->where('status','!=','Rejected')
                ->orderBy('id', 'asc')
                ->get()
                ->map(function ($entry) {
                    // Define mapping for module assignment
                    $mapping = [
                        'credit_id' => 'Credit',
                        'collection_id' => 'Collection',
                        'intake_id' => 'Intake',
                        'payment_id' => 'Payment',
                    ];

                    foreach ($mapping as $key => $value) {
                        if ($entry->$key) {
                            $entry->module = $value;
                            break;
                        }
                    }

                    // Default to "Others" if no module is matched
                    $entry->module = $entry->module ?? "Others";

                    return $entry;
                });

            // Add `deletable` attribute based on duplicates
            $uniqueEntries = [];
            $entries = $entries->map(function ($entry) use (&$uniqueEntries) {
                // Generate a unique key for duplicate detection
                $uniqueKey = null;

                if ($entry->credit_id) {
                    $uniqueKey = "credit_{$entry->credit_id}";
                } elseif ($entry->collection_id) {
                    $uniqueKey = "collection_{$entry->collection_id}";
                } elseif ($entry->intake_id) {
                    $uniqueKey = "intake_{$entry->intake_id}";
                } elseif ($entry->payment_id) {
                    $uniqueKey = "payment_{$entry->payment_id}";
                }

                // Check if this unique key already exists
                if ($uniqueKey && isset($uniqueEntries[$uniqueKey])) {
                    $entry->deletable = true;
                } else {
                    $entry->deletable = false;
                    if ($uniqueKey) {
                        $uniqueEntries[$uniqueKey] = true;
                    }
                }

                return $entry;
            });

        }

        $suppliers = Supplier::all();

        return view('customers.ledgers',compact('supplier_id','entries','suppliers'));
    }

    public function showDeletedSupplierLedgers()
    {
        $suppliers = Supplier::orderBy('name')->pluck('name', 'id');
        $selectedSupplier = request('supplier_id');

        $query = Ledger::onlyTrashed()
            ->whereNotNull('supplier_id')
            ->orderBy('deleted_at');

        if ($selectedSupplier) {
            $query->where('supplier_id', $selectedSupplier);
        }

        $deletedRecords = $query->get()
            ->groupBy(function($item) {
                return $item->deleted_at->format('Y-m-d');
            });

        return view('suppliers.deleted-suppliers', compact('deletedRecords', 'suppliers', 'selectedSupplier'));
    }
}
